<?php
// Allocate pembayaran_item rows with NULL kontrak_installment_id to unpaid installments.
// Usage:
//  php allocate_payments_to_installments.php <kontrak_id> [--apply --yes]

$argv = $_SERVER['argv'];
$kontrakId = isset($argv[1]) ? intval($argv[1]) : 0;
$apply = in_array('--apply', $argv);
$yes = in_array('--yes', $argv);
if (!$kontrakId) {
    echo "Usage: php allocate_payments_to_installments.php <kontrak_id> [--apply --yes]\n";
    exit(1);
}

$dbConfig = require __DIR__ . '/../config/db.php';
$dsn = $dbConfig['dsn'];
preg_match('/host=([^;]+);dbname=([^;]+)/', $dsn, $m);
$host = $m[1] ?? '127.0.0.1';
$dbname = $m[2] ?? 'cicilan';
$user = $dbConfig['username'] ?? 'root';
$pass = $dbConfig['password'] ?? '';

$mysqli = new mysqli($host, $user, $pass, $dbname);
if ($mysqli->connect_errno) {
    echo "DB connect error: " . $mysqli->connect_error . "\n";
    exit(1);
}
$mysqli->set_charset('utf8mb4');

// load installments
$insRes = $mysqli->query("SELECT id, periode, due_date, amount FROM kontrak_installment WHERE kontrak_id = " . intval($kontrakId) . " ORDER BY periode ASC");
$installments = [];
while ($r = $insRes->fetch_assoc()) $installments[] = $r;
if (empty($installments)) {
    echo "No installments for kontrak $kontrakId\n";
    exit(0);
}

// helper: current paid per installment
function paidForInstallment($mysqli, $insId) {
    $q = $mysqli->query("SELECT COALESCE(SUM(amount),0) as s FROM pembayaran_item WHERE kontrak_installment_id = " . intval($insId));
    $row = $q->fetch_assoc();
    return floatval($row['s']);
}

// gather pembayaran_item with NULL kontrak_installment_id for this kontrak
$sql = "SELECT pi.* FROM pembayaran_item pi JOIN pembayaran p ON p.id = pi.pembayaran_id WHERE p.kontrak_id = " . intval($kontrakId) . " AND (pi.kontrak_installment_id IS NULL OR pi.kontrak_installment_id = '') ORDER BY pi.id ASC";
$res = $mysqli->query($sql);
$orphans = [];
while ($r = $res->fetch_assoc()) $orphans[] = $r;

if (empty($orphans)) {
    echo "No orphan pembayaran_item rows to allocate for kontrak $kontrakId\n";
    exit(0);
}

echo "Found " . count($orphans) . " orphan pembayaran_item(s) for kontrak $kontrakId\n";

$plans = [];

foreach ($orphans as $item) {
    $remaining = floatval($item['amount']);
    foreach ($installments as $ins) {
        if ($remaining <= 0) break;
        $paid = paidForInstallment($mysqli, $ins['id']);
        $unpaid = floatval($ins['amount']) - $paid;
        if ($unpaid <= 0) continue;
        $alloc = min($unpaid, $remaining);
        $plans[] = [
            'pembayaran_item_id' => $item['id'],
            'pembayaran_id' => $item['pembayaran_id'],
            'original_amount' => floatval($item['amount']),
            'allocate_amount' => $alloc,
            'to_installment_id' => $ins['id'],
            'paid_at' => $item['paid_at'],
        ];
        $remaining -= $alloc;
    }
    if ($remaining > 0) {
        // leftover remains unallocated
        $plans[] = [
            'pembayaran_item_id' => $item['id'],
            'pembayaran_id' => $item['pembayaran_id'],
            'original_amount' => floatval($item['amount']),
            'allocate_amount' => 0,
            'to_installment_id' => null,
            'note' => 'leftover',
        ];
    }
}

// show plan summary
foreach ($plans as $p) {
    if ($p['allocate_amount'] > 0) {
        echo sprintf("Will allocate %.0f from pembayaran_item %s to installment %s (pembayaran %s)\n",
            $p['allocate_amount'],$p['pembayaran_item_id'],$p['to_installment_id'],$p['pembayaran_id']);
    }
}

if (!$apply) {
    echo "\nDry-run complete. To apply these changes run with --apply --yes\n";
    $mysqli->close();
    exit(0);
}

if ($apply && !$yes) {
    echo "Add --yes to confirm apply. Aborting.\n";
    $mysqli->close();
    exit(2);
}

// apply within transaction
$mysqli->begin_transaction();
try {
    foreach ($plans as $p) {
        if ($p['allocate_amount'] <= 0) continue;
        $piId = intval($p['pembayaran_item_id']);
        $alloc = floatval($p['allocate_amount']);
        $toIns = $p['to_installment_id'];

        // fetch current amount to decide split
        $q = $mysqli->query("SELECT amount FROM pembayaran_item WHERE id = $piId FOR UPDATE");
        $row = $q->fetch_assoc();
        $curr = floatval($row['amount']);
        if (abs($curr - $alloc) < 0.01) {
            // exact fit: update kontrak_installment_id
            $upd = $mysqli->query("UPDATE pembayaran_item SET kontrak_installment_id = " . intval($toIns) . " WHERE id = $piId");
            if ($mysqli->errno) throw new Exception($mysqli->error);
        } elseif ($curr > $alloc) {
            // split: reduce original to alloc, create leftover row
            $upd = $mysqli->query("UPDATE pembayaran_item SET amount = " . $alloc . ", kontrak_installment_id = " . intval($toIns) . " WHERE id = $piId");
            if ($mysqli->errno) throw new Exception($mysqli->error);
            $left = $curr - $alloc;
            $paidAtRaw = $p['paid_at'] ?? date('Y-m-d H:i:s');
            $paidAtEsc = "'" . $mysqli->real_escape_string($paidAtRaw) . "'";
            $insSql = sprintf("INSERT INTO pembayaran_item (pembayaran_id, kontrak_installment_id, amount, paid_at, note) VALUES (%d, NULL, %f, %s, NULL)",
                intval($p['pembayaran_id']), $left, $paidAtEsc);
            $mysqli->query($insSql);
            if ($mysqli->errno) throw new Exception($mysqli->error);
        } else {
            // curr < alloc shouldn't happen, skip
            throw new Exception("Current amount < allocate for pembayaran_item $piId");
        }
    }

    $mysqli->commit();
    echo "Applied allocations successfully.\n";
} catch (Exception $e) {
    $mysqli->rollback();
    echo "Error applying allocations: " . $e->getMessage() . "\n";
    $mysqli->close();
    exit(3);
}

$mysqli->close();
exit(0);
