<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;
use yii\web\IdentityInterface;

/**
 * User ActiveRecord for authentication and user management.
 *
 * Columns: id, username, password_hash, auth_key, role, created_at, updated_at
 */
class User extends ActiveRecord implements IdentityInterface
{
    public static function tableName()
    {
        return '{{%user}}';
    }

    public function rules()
    {
        return [
            [['username', 'role'], 'required'],
            ['password_hash', 'required', 'on' => 'create'],
            ['username', 'string', 'max' => 255],
            ['username', 'unique'],
            ['password_hash', 'string', 'min' => 3],
            ['role', 'string', 'max' => 64],
        ];
    }

    public function attributeLabels()
    {
        return [
            'username' => 'Username',
            'password_hash' => 'Password',
            'role' => 'Role',
        ];
    }

    public static function findIdentity($id)
    {
        return static::findOne($id);
    }

    public static function findIdentityByAccessToken($token, $type = null)
    {
        return static::findOne(['access_token' => $token]);
    }

    public static function findByUsername($username)
    {
        return static::findOne(['username' => $username]);
    }

    public function getId()
    {
        return $this->getPrimaryKey();
    }

    public function getAuthKey()
    {
        return $this->auth_key;
    }

    public function validateAuthKey($authKey)
    {
        return $this->getAuthKey() === $authKey;
    }

    public function validatePassword($password)
    {
        return Yii::$app->security->validatePassword($password, $this->password_hash);
    }

    public function setPassword($password)
    {
        $this->password_hash = Yii::$app->security->generatePasswordHash($password);
    }

    public function generateAuthKey()
    {
        $this->auth_key = Yii::$app->security->generateRandomString();
    }

    /**
     * Check whether user has a permission by name using role_permission tables.
     */
    public function hasPermission($permissionName)
    {
        if (!$this->role) {
            return false;
        }
        $role = Role::findOne(['name' => $this->role]);
        if (!$role) {
            return false;
        }
        return (bool) $role->getPermissions()->andWhere(['name' => $permissionName])->exists();
    }
}
