<?php

namespace app\models;

use Yii;
use yii\base\Model;

/**
 * Simple Setting helper that stores key/value pairs in DB table `setting`.
 * If table does not exist, methods fall back to defaults or create the table on set().
 */
class Setting extends Model
{
    /** Get a value by key or default */
    public static function get(string $key, $default = null)
    {
        try {
            $val = Yii::$app->db->createCommand('SELECT `value` FROM `setting` WHERE `key`=:k')
                ->bindValue(':k', $key)
                ->queryScalar();
            return $val === false ? $default : $val;
        } catch (\Exception $e) {
            return $default;
        }
    }

    /** Set a key value; creates table if missing */
    public static function set(string $key, $value): bool
    {
        $db = Yii::$app->db;
        try {
            // ensure table exists
            $db->createCommand(<<<'SQL'
                CREATE TABLE IF NOT EXISTS `setting` (
                  `key` varchar(191) NOT NULL,
                  `value` text,
                  PRIMARY KEY (`key`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
            SQL
            )->execute();

            $exists = $db->createCommand('SELECT COUNT(*) FROM `setting` WHERE `key`=:k')
                ->bindValue(':k', $key)
                ->queryScalar();
            if ($exists > 0) {
                $db->createCommand()->update('setting', ['value' => (string)$value], ['key' => $key])->execute();
            } else {
                $db->createCommand()->insert('setting', ['key' => $key, 'value' => (string)$value])->execute();
            }
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }
}
