<?php

namespace app\models;

use Yii;
use app\models\Setting;

/**
 * This is the model class for table "barang".
 *
 * @property string $kode_barang
 * @property string $nama
 * @property string|null $deskripsi
 * @property float $harga
 * @property string|null $satuan
 * @property int|null $stok
 * @property string|null $foto
 * @property string|null $kategori
 */
class Barang extends \yii\db\ActiveRecord
{
    /**
     * Normalize harga input before validation (remove dots, currency chars)
     */
    public function beforeValidate()
    {
        if (!parent::beforeValidate()) {
            return false;
        }
        if ($this->hasAttribute('harga') && $this->harga !== null) {
            // remove non-digit characters
            $this->harga = preg_replace('/[^0-9]/', '', (string)$this->harga);
            if ($this->harga === '') {
                $this->harga = null;
            }
        }
        return true;
    }

    /**
     * Generate kode for barang using prefix from settings or default BRG-
     * This uses the record id (autoincrement) so should be called after initial save if id is not present.
     * @param string|null $prefix
     * @return string
     */
    public function generateKode(?string $prefix = null): string
    {
        if ($prefix === null) {
            $prefix = Setting::get('prefix_barang', 'BRG-');
        }
        // Find the latest kode_barang with the same prefix and increment its numeric suffix
        $query = (new \yii\db\Query())
            ->select('kode_barang')
            ->from(self::tableName())
            ->where(['like', 'kode_barang', $prefix . '%', false])
            ->orderBy(['kode_barang' => SORT_DESC])
            ->limit(1);
        $row = $query->one();
        $lastKode = $row['kode_barang'] ?? null;
        if ($lastKode) {
            // extract trailing digits
            if (preg_match('/(\d+)$/', $lastKode, $m)) {
                $num = (int)$m[1] + 1;
            } else {
                $num = 1;
            }
        } else {
            $num = 1;
        }
        return $prefix . str_pad((string)$num, 4, '0', STR_PAD_LEFT);
    }
    /** @var \yii\web\UploadedFile|null */
    public $fotoFile;
    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'barang';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['deskripsi', 'satuan', 'foto'], 'default', 'value' => null],
            [['stok'], 'default', 'value' => 0],
            [['kategori'], 'default', 'value' => 'Umum'],
            [['nama', 'harga'], 'required'],
            [['deskripsi'], 'string'],
            [['harga'], 'number'],
            [['stok'], 'integer'],
            [['kode_barang'], 'string', 'max' => 10],
            [['nama', 'foto'], 'string', 'max' => 255],
            [['fotoFile'], 'file', 'skipOnEmpty' => true, 'extensions' => 'png, jpg, jpeg, gif', 'maxSize' => 5242880],
            [['satuan'], 'string', 'max' => 50],
            [['kategori'], 'string', 'max' => 100],
            [['kode_barang'], 'unique'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'kode_barang' => 'Kode Barang',
            'nama' => 'Nama',
            'deskripsi' => 'Deskripsi',
            'harga' => 'Harga',
            'satuan' => 'Satuan',
            'stok' => 'Stok',
            'foto' => 'Foto',
            'kategori' => 'Kategori',
        ];
    }

    /**
     * Relation to Kategori model. Maps `kategori` (string) in barang
     * to `nama` in kategori table.
     * @return \yii\db\ActiveQuery
     */
    public function getKategoriModel()
    {
        return $this->hasOne(Kategori::class, ['nama' => 'kategori']);
    }

}
