<?php

use yii\db\Migration;

/**
 * Migration to replace `barang.id` primary key with `barang.kode_barang` as primary key.
 *
 * What it does (up):
 *  - Finds all foreign key constraints that reference `barang(id)` and for each:
 *      - Adds a temporary varchar column in the referencing table
 *      - Copies kode_barang into that temp column by joining on barang.id
 *      - Drops the FK constraint
 *      - Drops the old integer FK column
 *      - Renames the temp column back to the original FK column name
 *  - Alters `barang.kode_barang` to NOT NULL
 *  - Drops `barang` primary key on `id`, drops `id` column
 *  - Adds primary key on `kode_barang`
 *  - Recreates foreign keys from referencing tables to `barang(kode_barang)`
 *
 * Notes:
 *  - This migration assumes `kode_barang` already contains unique values for all rows.
 *  - It attempts to detect referencing tables dynamically via information_schema.
 *  - The migration is destructive (drops `id`). Make a DB backup before applying.
 */
class m251210_120000_convert_barang_id_to_kode_pk extends Migration
{
    public function safeUp()
    {
        $dbName = $this->db->createCommand('SELECT DATABASE()')->queryScalar();
        if (!$dbName) {
            throw new \Exception('Unable to detect database name.');
        }

        // Get column type/length for kode_barang
        $col = $this->db->createCommand("SELECT COLUMN_TYPE, CHARACTER_MAXIMUM_LENGTH
            FROM information_schema.COLUMNS
            WHERE TABLE_SCHEMA = :db AND TABLE_NAME = 'barang' AND COLUMN_NAME = 'kode_barang'", [':db' => $dbName])->queryOne();

        if (!$col) {
            throw new \Exception('Column `barang.kode_barang` not found.');
        }
        $columnType = $col['COLUMN_TYPE'];
        $charMax = $col['CHARACTER_MAXIMUM_LENGTH'] ?: 255;

        // Find all foreign keys referencing barang(id)
        $refs = $this->db->createCommand("SELECT TABLE_NAME, COLUMN_NAME, CONSTRAINT_NAME
            FROM information_schema.KEY_COLUMN_USAGE
            WHERE REFERENCED_TABLE_SCHEMA = :db
              AND REFERENCED_TABLE_NAME = 'barang'
              AND REFERENCED_COLUMN_NAME = 'id'", [':db' => $dbName])->queryAll();

        $refsGrouped = [];
        foreach ($refs as $r) {
            $refsGrouped[$r['TABLE_NAME']][] = $r;
        }

        // Keep list to recreate FKs later
        $fkToCreate = [];

        // For each referencing table, convert FK column to store kode_barang values
        foreach ($refsGrouped as $table => $constraints) {
            foreach ($constraints as $cons) {
                $colName = $cons['COLUMN_NAME'];
                $constraintName = $cons['CONSTRAINT_NAME'];
                $tmpCol = $colName . '_kode_tmp';

                // Add temporary column
                $this->addColumn($table, $tmpCol, $this->string($charMax)->null());

                // Copy kode_barang into temp column using join on id
                $this->execute("UPDATE `{$table}` t JOIN `barang` b ON b.id = t.`{$colName}` SET t.`{$tmpCol}` = b.`kode_barang`");

                // Drop foreign key constraint (if exists)
                $this->execute("ALTER TABLE `{$table}` DROP FOREIGN KEY `{$constraintName}`");

                // Drop the old column (we will rename temp to old name)
                $this->dropColumn($table, $colName);

                // Rename temp back to original column name
                $this->renameColumn($table, $tmpCol, $colName);

                // Record to recreate FK after barang PK switched
                $fkToCreate[] = [
                    'table' => $table,
                    'column' => $colName,
                ];
            }
        }

        // Ensure kode_barang is NOT NULL and has same column type
        $this->execute("ALTER TABLE `barang` MODIFY `kode_barang` {$columnType} NOT NULL");

        // Drop primary key (assumes it was on `id`) and drop id column
        // First remove AUTO_INCREMENT from id so MySQL allows dropping the PK
        $this->execute("ALTER TABLE `barang` MODIFY `id` int(11) NOT NULL");
        // Note: if primary key is composite or different, this may need manual adjustment
        $this->execute("ALTER TABLE `barang` DROP PRIMARY KEY");
        $this->dropColumn('barang', 'id');

        // Add primary key on kode_barang
        // If a UNIQUE index exists on kode_barang, drop it first to avoid duplicate index name error
        $this->execute("ALTER TABLE `barang` DROP INDEX `kode_barang`");
        $this->execute("ALTER TABLE `barang` ADD PRIMARY KEY (`kode_barang`)");

        // Recreate foreign keys pointing to barang(kode_barang)
        foreach ($fkToCreate as $fk) {
            $table = $fk['table'];
            $colName = $fk['column'];
            $fkName = "fk_{$table}_{$colName}_barang_kode";
            // Add index (if not exists) then add FK
            $this->createIndex('idx_'.$table.'_'.$colName, $table, $colName);
            $this->addForeignKey($fkName, $table, $colName, 'barang', 'kode_barang', 'CASCADE', 'CASCADE');
        }

        return true;
    }

    public function safeDown()
    {
        echo "m251210_120000_convert_barang_id_to_kode_pk cannot be reverted automatically.\n";
        return false;
    }
}
