<?php

namespace app\controllers;

use Yii;
use app\models\User;
use yii\data\ActiveDataProvider;
use yii\web\Controller;
use yii\filters\VerbFilter;
use yii\filters\AccessControl;
use yii\web\NotFoundHttpException;

class UserController extends Controller
{
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'only' => ['index','create','update','delete','view'],
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function ($rule, $action) {
                            return Yii::$app->user->identity && Yii::$app->user->identity->role === 'admin';
                        }
                    ],
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => ['delete' => ['POST']],
            ],
        ];
    }

    public function actionIndex()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => User::find()->orderBy(['id' => SORT_DESC]),
            'pagination' => ['pageSize' => 20],
        ]);

        return $this->render('index', ['dataProvider' => $dataProvider]);
    }

    public function actionView($id)
    {
        $model = $this->findModel($id);
        if (Yii::$app->request->isAjax) {
            return $this->renderAjax('view', ['model' => $model]);
        }
        return $this->render('view', ['model' => $model]);
    }

    public function actionCreate()
    {
        $model = new User();

        $roles = \app\models\Role::find()->select(['name','id'])->indexBy('name')->column();
        $model->scenario = 'create';
        // support AJAX form rendering and submission for modal quick-create
        if (Yii::$app->request->isAjax) {
            if ($model->load(Yii::$app->request->post())) {
                if (!empty($model->password_hash)) {
                    $model->setPassword($model->password_hash);
                }
                $model->generateAuthKey();
                $model->created_at = time();
                $model->updated_at = time();
                if ($model->save()) {
                    Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;
                    return ['success' => true, 'message' => 'User berhasil dibuat.'];
                }
                // return rendered form with validation errors
                return $this->renderAjax('create', ['model' => $model, 'roles' => $roles]);
            }
            return $this->renderAjax('create', ['model' => $model, 'roles' => $roles]);
        }

        if ($model->load(Yii::$app->request->post())) {
            if (!empty($model->password_hash)) {
                $model->setPassword($model->password_hash);
            }
            $model->generateAuthKey();
            $model->created_at = time();
            $model->updated_at = time();
            if ($model->save()) {
                return $this->redirect(['view', 'id' => $model->id]);
            }
        }

        return $this->render('create', ['model' => $model, 'roles' => $roles]);
    }

    public function actionUpdate($id)
    {
        $model = $this->findModel($id);

        // allow setting a new plain password via attribute 'password_hash' in form
        $plainPassword = $model->password_hash;
        $roles = \app\models\Role::find()->select(['name','id'])->indexBy('name')->column();
        if (Yii::$app->request->isAjax) {
            if ($model->load(Yii::$app->request->post())) {
                if (!empty($model->password_hash) && $model->password_hash !== $plainPassword) {
                    $model->setPassword($model->password_hash);
                } else {
                    // keep existing hash
                    $model->password_hash = $plainPassword;
                }
                $model->updated_at = time();
                if ($model->save()) {
                    Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;
                    return ['success' => true, 'message' => 'User berhasil diperbarui.'];
                }
                return $this->renderAjax('update', ['model' => $model, 'roles' => $roles]);
            }
            // blank out password field in form
            $model->password_hash = '';
            return $this->renderAjax('update', ['model' => $model, 'roles' => $roles]);
        }

        if ($model->load(Yii::$app->request->post())) {
            if (!empty($model->password_hash) && $model->password_hash !== $plainPassword) {
                $model->setPassword($model->password_hash);
            } else {
                $model->password_hash = $plainPassword;
            }
            $model->updated_at = time();
            if ($model->save()) {
                return $this->redirect(['view', 'id' => $model->id]);
            }
        }

        // blank out password field in form
        $model->password_hash = '';
        return $this->render('update', ['model' => $model, 'roles' => $roles]);
    }

    public function actionDelete($id)
    {
        $model = $this->findModel($id);
        // prevent deleting last admin
        if ($model->role === 'admin') {
            $count = User::find()->where(['role' => 'admin'])->andWhere(['<>', 'id', $model->id])->count();
            if ($count <= 0) {
                if (Yii::$app->request->isAjax) {
                    Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;
                    return ['success' => false, 'message' => 'Tidak dapat menghapus akun admin terakhir.'];
                }
                Yii::$app->session->setFlash('error', 'Tidak dapat menghapus akun admin terakhir.');
                return $this->redirect(['index']);
            }
        }
        $model->delete();
        if (Yii::$app->request->isAjax) {
            Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;
            return ['success' => true, 'message' => 'User telah dihapus.'];
        }
        return $this->redirect(['index']);
    }

    protected function findModel($id)
    {
        if (($model = User::findOne($id)) !== null) {
            return $model;
        }
        throw new NotFoundHttpException('User not found.');
    }
}
