<?php
namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\web\UploadedFile;
use app\models\LogoForm;
use app\models\Setting;
use yii\helpers\Url;
use yii\web\BadRequestHttpException;
use yii\web\ForbiddenHttpException;

class SettingController extends Controller
{
    public function actionIndex()
    {
        return $this->render('index');
    }

    public function actionLogo()
    {
        $model = new LogoForm();
        if (Yii::$app->request->isPost) {
            $model->imageFile = UploadedFile::getInstance($model, 'imageFile');
            if ($model->upload()) {
                Yii::$app->session->setFlash('success', 'Logo berhasil diunggah.');
                return $this->refresh();
            } else {
                Yii::$app->session->setFlash('error', 'Gagal mengunggah logo.');
            }
        }
        return $this->render('logo', ['model' => $model]);
    }

    public function actionPrefix()
    {
        $prefixBarang = Setting::get('prefix_barang', 'BRG-');
        $prefixPelanggan = Setting::get('prefix_pelanggan', 'PLG-');
        $prefixPayment = Setting::get('prefix_payment', 'PAY-');
        $prefixKontrak = Setting::get('prefix_kontrak', 'KTR-');
        return $this->render('prefix', [
            'prefixBarang' => $prefixBarang,
            'prefixPelanggan' => $prefixPelanggan,
            'prefixPayment' => $prefixPayment,
            'prefixKontrak' => $prefixKontrak,
        ]);
    }

    public function actionSave()
    {
        if (!Yii::$app->request->isPost) {
            throw new BadRequestHttpException('Invalid request');
        }
        $pb = Yii::$app->request->post('prefix_barang');
        $pp = Yii::$app->request->post('prefix_pelanggan');
        $ppay = Yii::$app->request->post('prefix_payment');
        $pktr = Yii::$app->request->post('prefix_kontrak');
        Setting::set('prefix_barang', $pb ?: 'BRG-');
        Setting::set('prefix_pelanggan', $pp ?: 'PLG-');
        Setting::set('prefix_payment', $ppay ?: 'PAY-');
        Setting::set('prefix_kontrak', $pktr ?: 'KTR-');
        Yii::$app->session->setFlash('success','Prefix berhasil disimpan.');
        return $this->redirect(Url::to(['prefix']));
    }

    public function actionRegenerate()
    {
        // regenerate kode for barang and pelanggan using current prefixes
        $db = Yii::$app->db;
        $pb = Setting::get('prefix_barang', 'BRG-');
        $pp = Setting::get('prefix_pelanggan', 'PLG-');

        // barang
        $rows = $db->createCommand('SELECT id FROM barang')->queryAll();
        foreach ($rows as $r) {
            $id = $r['id'];
            $kode = $pb . str_pad($id, 4, '0', STR_PAD_LEFT);
            $db->createCommand()->update('barang', ['kode_barang' => $kode], ['id' => $id])->execute();
        }

        // pelanggan
        try {
            $rows = $db->createCommand('SELECT id FROM pelanggan')->queryAll();
            foreach ($rows as $r) {
                $id = $r['id'];
                $kode = $pp . str_pad($id, 4, '0', STR_PAD_LEFT);
                $db->createCommand()->update('pelanggan', ['kode_pelanggan' => $kode], ['id' => $id])->execute();
            }
        } catch (\Exception $e) {
            // table has no `id` column (converted to kode_pelanggan PK).
            // regenerate sequentially based on current ordering.
            $rows = $db->createCommand('SELECT kode_pelanggan FROM pelanggan ORDER BY created_at, kode_pelanggan')->queryAll();
            $i = 1;
            foreach ($rows as $r) {
                $kode = $pp . str_pad($i, 4, '0', STR_PAD_LEFT);
                $db->createCommand()->update('pelanggan', ['kode_pelanggan' => $kode], ['kode_pelanggan' => $r['kode_pelanggan']])->execute();
                $i++;
            }
        }

        Yii::$app->session->setFlash('success','Regenerasi kode selesai.');
        return $this->redirect(['prefix']);
    }

    public function actionDatabase()
    {
        return $this->render('database');
    }

    public function actionUser()
    {
        // Render role management index here so role CRUD is accessible under setting/user
        // only allow admin users
        if (Yii::$app->user->isGuest || Yii::$app->user->identity->role !== 'admin') {
            return $this->render('user');
        }

        // prepare data providers for embedded management UIs
        $userProvider = new \yii\data\ActiveDataProvider([
            'query' => \app\models\User::find(),
            'pagination' => ['pageSize' => 20],
        ]);

        $roleProvider = new \yii\data\ActiveDataProvider([
            'query' => \app\models\Role::find(),
            'pagination' => ['pageSize' => 20],
        ]);

        $permissionProvider = new \yii\data\ActiveDataProvider([
            'query' => \app\models\Permission::find(),
            'pagination' => ['pageSize' => 20],
        ]);

        return $this->render('user', [
            'userProvider' => $userProvider,
            'roleProvider' => $roleProvider,
            'permissionProvider' => $permissionProvider,
        ]);
    }

    // Provide role CRUD under setting namespace to keep management centralized
    public function actionRoleView($id)
    {
        $this->checkAdmin();
        $model = \app\models\Role::findOne($id);
        if (!$model) throw new \yii\web\NotFoundHttpException('Role not found.');
        $allPermissions = \app\models\Permission::find()->all();
        return $this->render('@app/views/role/view', ['model' => $model, 'allPermissions' => $allPermissions]);
    }

    public function actionRoleCreate()
    {
        $this->checkAdmin();
        $model = new \app\models\Role();
        $permissions = \app\models\Permission::find()->select(['name','id'])->indexBy('id')->column();
        if ($model->load(Yii::$app->request->post())) {
            if ($model->save()) {
                // save permissions
                $posted = Yii::$app->request->post('permissions', []);
                foreach ($posted as $pid) {
                    Yii::$app->db->createCommand()->insert('{{%role_permission}}', ['role_id' => $model->id, 'permission_id' => $pid])->execute();
                }
                return $this->redirect(['user']);
            }
        }
        return $this->render('@app/views/role/create', ['model' => $model, 'permissions' => $permissions]);
    }

    public function actionRoleUpdate($id)
    {
        $this->checkAdmin();
        $model = \app\models\Role::findOne($id);
        if (!$model) throw new \yii\web\NotFoundHttpException('Role not found.');
        $permissions = \app\models\Permission::find()->select(['name','id'])->indexBy('id')->column();
        if ($model->load(Yii::$app->request->post())) {
            if ($model->save()) {
                // save permissions
                Yii::$app->db->createCommand()->delete('{{%role_permission}}', ['role_id' => $model->id])->execute();
                $posted = Yii::$app->request->post('permissions', []);
                foreach ($posted as $pid) {
                    Yii::$app->db->createCommand()->insert('{{%role_permission}}', ['role_id' => $model->id, 'permission_id' => $pid])->execute();
                }
                return $this->redirect(['user']);
            }
        }
        return $this->render('@app/views/role/update', ['model' => $model, 'permissions' => $permissions]);
    }

    public function actionRoleDelete($id)
    {
        $this->checkAdmin();
        $model = \app\models\Role::findOne($id);
        if ($model) $model->delete();
        return $this->redirect(['user']);
    }

    protected function checkAdmin()
    {
        // allow if user has manageSettings or manageUsers permission, or role === 'admin'
        if (Yii::$app->user->isGuest) {
            throw new ForbiddenHttpException('Only admins allowed.');
        }
        $user = Yii::$app->user->identity;
        if ($user->role === 'admin') return;
        if ($user->hasPermission('manageSettings') || $user->hasPermission('manageUsers')) return;
        throw new ForbiddenHttpException('Only admins allowed.');
    }

    public function actionOthers()
    {
        return $this->render('others');
    }
}

