<?php

namespace app\controllers;

use Yii;
use app\models\Role;
use app\models\Permission;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use yii\web\NotFoundHttpException;
use yii\data\ActiveDataProvider;

class RoleController extends Controller
{
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'only' => ['index','create','update','delete','view'],
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function () {
                            return Yii::$app->user->identity && Yii::$app->user->identity->role === 'admin';
                        }
                    ],
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => ['delete' => ['POST']],
            ],
        ];
    }

    public function actionIndex()
    {
        $dataProvider = new ActiveDataProvider(['query' => Role::find()]);
        return $this->render('index', ['dataProvider' => $dataProvider]);
    }

    public function actionView($id)
    {
        $model = $this->findModel($id);
        $allPermissions = Permission::find()->all();
        if (Yii::$app->request->isAjax) {
            return $this->renderAjax('view', ['model' => $model, 'allPermissions' => $allPermissions]);
        }
        return $this->render('view', ['model' => $model, 'allPermissions' => $allPermissions]);
    }

    public function actionCreate()
    {
        $model = new Role();
        // support AJAX modal create
        if (Yii::$app->request->isAjax) {
            if ($model->load(Yii::$app->request->post())) {
                if ($model->save()) {
                    // handle matrix-style permissions if provided
                    $matrix = Yii::$app->request->post('matrix', null);
                    if ($matrix !== null) {
                        $this->saveMatrixPermissions($model, $matrix);
                    } else {
                        $this->savePermissions($model);
                    }
                    Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;
                    return ['success' => true, 'message' => 'Role berhasil dibuat.'];
                }
                // return rendered form with validation errors
                $permissions = Permission::find()->select(['name','id'])->indexBy('id')->column();
                return $this->renderAjax('create', ['model' => $model, 'permissions' => $permissions]);
            }
            $permissions = Permission::find()->select(['name','id'])->indexBy('id')->column();
            return $this->renderAjax('create', ['model' => $model, 'permissions' => $permissions]);
        }

        if ($model->load(Yii::$app->request->post())) {
            if ($model->save()) {
                $matrix = Yii::$app->request->post('matrix', null);
                if ($matrix !== null) {
                    $this->saveMatrixPermissions($model, $matrix);
                } else {
                    $this->savePermissions($model);
                }
                return $this->redirect(['view', 'id' => $model->id]);
            }
        }
        $permissions = Permission::find()->select(['name','id'])->indexBy('id')->column();
        return $this->render('create', ['model' => $model, 'permissions' => $permissions]);
    }

    public function actionUpdate($id)
    {
        $model = $this->findModel($id);
        if (Yii::$app->request->isAjax) {
            if ($model->load(Yii::$app->request->post())) {
                if ($model->save()) {
                    $matrix = Yii::$app->request->post('matrix', null);
                    if ($matrix !== null) {
                        $this->saveMatrixPermissions($model, $matrix);
                    } else {
                        $this->savePermissions($model);
                    }
                    Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;
                    return ['success' => true, 'message' => 'Role berhasil diperbarui.'];
                }
                $permissions = Permission::find()->select(['name','id'])->indexBy('id')->column();
                return $this->renderAjax('update', ['model' => $model, 'permissions' => $permissions]);
            }
            $permissions = Permission::find()->select(['name','id'])->indexBy('id')->column();
            return $this->renderAjax('update', ['model' => $model, 'permissions' => $permissions]);
        }

        if ($model->load(Yii::$app->request->post())) {
            if ($model->save()) {
                $matrix = Yii::$app->request->post('matrix', null);
                if ($matrix !== null) {
                    $this->saveMatrixPermissions($model, $matrix);
                } else {
                    $this->savePermissions($model);
                }
                return $this->redirect(['view', 'id' => $model->id]);
            }
        }
        $permissions = Permission::find()->select(['name','id'])->indexBy('id')->column();
        return $this->render('update', ['model' => $model, 'permissions' => $permissions]);
    }

    protected function saveMatrixPermissions(Role $role, array $matrix)
    {
        $permNames = [];
        foreach ($matrix as $menu => $actions) {
            foreach ($actions as $action => $val) {
                if ($val) $permNames[] = $menu . '.' . $action;
            }
        }
        // ensure permissions exist, collect ids
        $ids = [];
        foreach ($permNames as $name) {
            $perm = Permission::find()->where(['name' => $name])->one();
            if (!$perm) {
                $perm = new Permission();
                $perm->name = $name;
                $perm->description = $name;
                $perm->save(false);
            }
            $ids[] = $perm->id;
        }
        // delete existing mappings
        \Yii::$app->db->createCommand()->delete('{{%role_permission}}', ['role_id' => $role->id])->execute();
        foreach ($ids as $pid) {
            \Yii::$app->db->createCommand()->insert('{{%role_permission}}', ['role_id' => $role->id, 'permission_id' => $pid])->execute();
        }
    }

    public function actionDelete($id)
    {
        $model = $this->findModel($id);
        // prevent deleting role if users still assigned to it
        $count = \app\models\User::find()->where(['role' => $model->name])->count();
        if ($count > 0) {
            if (Yii::$app->request->isAjax) {
                Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;
                return ['success' => false, 'message' => 'Role ini masih digunakan oleh user. Pindahkan atau hapus user terlebih dahulu.'];
            }
            Yii::$app->session->setFlash('error', 'Role ini masih digunakan oleh user. Pindahkan atau hapus user terlebih dahulu.');
            return $this->redirect(['index']);
        }
        $model->delete();
        if (Yii::$app->request->isAjax) {
            Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;
            return ['success' => true, 'message' => 'Role telah dihapus.'];
        }
        return $this->redirect(['index']);
    }

    protected function findModel($id)
    {
        if (($model = Role::findOne($id)) !== null) {
            return $model;
        }
        throw new NotFoundHttpException('Role not found.');
    }

    protected function savePermissions(Role $role)
    {
        $posted = Yii::$app->request->post('permissions', []);
        // clear existing
        \Yii::$app->db->createCommand()->delete('{{%role_permission}}', ['role_id' => $role->id])->execute();
        foreach ($posted as $pid) {
            \Yii::$app->db->createCommand()->insert('{{%role_permission}}', ['role_id' => $role->id, 'permission_id' => $pid])->execute();
        }
    }
}
