<?php
namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\web\Response;
use yii\filters\VerbFilter;
use yii\web\ForbiddenHttpException;
use app\models\Pelanggan;
use app\models\Kontrak;
use app\models\KontrakInstallment;
use app\models\Pembayaran;
use app\models\PembayaranItem;
use app\models\Setting;

class PembayaranController extends Controller
{
    public function behaviors()
    {
        return [
            'verbs'=>[ 'class'=>VerbFilter::class, 'actions'=>['create'=>['GET','POST'],'delete'=>['POST']] ],
        ];
    }

    public function beforeAction($action)
    {
        if (!parent::beforeAction($action)) return false;
        $map = [
            'index' => 'pembayaran.view',
            'view' => 'pembayaran.view',
            'create' => 'pembayaran.create',
            'delete' => 'pembayaran.delete',
            'history' => 'pembayaran.view',
            'get-installments' => 'pembayaran.view',
            'kontrak-for-pelanggan' => 'pembayaran.view',
        ];
        $id = $action->id;
        $required = $map[$id] ?? null;
        if ($required) {
            if (Yii::$app->user->isGuest || !Yii::$app->user->identity->hasPermission($required)) {
                throw new ForbiddenHttpException('Anda tidak memiliki izin untuk mengakses halaman ini.');
            }
        }
        return true;
    }

    public function actionDelete($id)
    {
        $request = Yii::$app->request;
        $model = Pembayaran::findOne($id);
        if (!$model) {
            if ($request->isAjax) {
                Yii::$app->response->format = Response::FORMAT_JSON;
                return ['success'=>false,'message'=>'Pembayaran tidak ditemukan'];
            }
            throw new \yii\web\NotFoundHttpException('Pembayaran tidak ditemukan');
        }

        $db = Yii::$app->db;
        $tx = $db->beginTransaction();
        try {
            PembayaranItem::deleteAll(['pembayaran_id'=>$model->id]);
            $model->delete();
            $tx->commit();
            if ($request->isAjax) {
                Yii::$app->response->format = Response::FORMAT_JSON;
                return ['success'=>true];
            }
            return $this->redirect($request->referrer ?: ['index']);
        } catch (\Throwable $e) {
            $tx->rollBack();
            Yii::error('Delete pembayaran error: ' . $e->getMessage(), __METHOD__);
            if ($request->isAjax) {
                Yii::$app->response->format = Response::FORMAT_JSON;
                return ['success'=>false,'message'=>$e->getMessage()];
            }
            throw $e;
        }
    }

    public function actionIndex()
    {
        $request = Yii::$app->request;
        $q = trim($request->get('q',''));
        $from = $request->get('from','');
        $to = $request->get('to','');

        $query = Pembayaran::find()->alias('p')
            ->leftJoin('pelanggan pl', "p.pelanggan_id = pl.kode_pelanggan")
            ->leftJoin('kontrak k', "p.kontrak_id = k.id");

        if ($q !== '') {
            $query->andWhere(['or',
                ['like','p.nomor',$q],
                ['like','pl.nama',$q],
                ['like','pl.no_ktp',$q],
                ['like','pl.no_whatsapp',$q],
                ['like','k.nomor',$q]
            ]);
        }

        // normalize date filters: treat 'from' as start of day and 'to' as end of day
        if ($from !== '' && $to !== '') {
            $fromDt = $from . ' 00:00:00';
            $toDt = $to . ' 23:59:59';
            $query->andWhere(['between','p.created_at',$fromDt,$toDt]);
        } elseif ($from !== '') {
            $fromDt = $from . ' 00:00:00';
            $query->andWhere(['>=','p.created_at',$fromDt]);
        } elseif ($to !== '') {
            $toDt = $to . ' 23:59:59';
            $query->andWhere(['<=','p.created_at',$toDt]);
        } else {
            $today = (new \DateTime('now', new \DateTimeZone(Yii::$app->timeZone)))->format('Y-m-d');
            $query->andWhere(['like','p.created_at',$today]);
        }

        $countQuery = clone $query;
        $pagination = new \yii\data\Pagination(['totalCount'=>$countQuery->count(),'pageSize'=>15]);
        $models = $query->orderBy(['p.created_at'=>SORT_DESC])->offset($pagination->offset)->limit($pagination->limit)->all();

        // If AJAX request (pagination or filter), return only the list partial
        if (Yii::$app->request->isAjax) {
            return $this->renderPartial('_list',['models'=>$models,'pagination'=>$pagination,'q'=>$q,'from'=>$from,'to'=>$to]);
        }

        return $this->render('index',['models'=>$models,'pagination'=>$pagination,'q'=>$q,'from'=>$from,'to'=>$to]);
    }

    public function actionCreate()
    {
        $request = Yii::$app->request;
        if ($request->isPost) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            $post = $request->post();
            // generate nomor pembayaran
            $prefix = Setting::get('pembayaran_prefix','PMB');
            $today = (new \DateTime('now', new \DateTimeZone(Yii::$app->timeZone)))->format('Ymd');
            $seqKey = 'pembayaran_seq_'.$today;
            $seq = intval(Setting::get($seqKey,0)) + 1;
            Setting::set($seqKey,$seq);
            $nomor = sprintf('%s-%s-%04d', $prefix, (new \DateTime('now', new \DateTimeZone(Yii::$app->timeZone)))->format('ymd'), $seq);

            $payments = $post['payments'] ?? [];
            // compute total without persisting yet
            $totalPaid = 0;
            $normalizedPayments = [];
            foreach ($payments as $pay) {
                $amt = floatval($pay['amount'] ?? 0);
                if ($amt <= 0) continue;
                $kId = $pay['kontrak_installment_id'] ?? ($pay['installment_id'] ?? null);
                $normalizedPayments[] = [
                    'kontrak_installment_id' => ($kId === '' || $kId === null) ? null : intval($kId),
                    'amount' => $amt,
                    'paid_at' => $pay['paid_at'] ?? (new \DateTime('now', new \DateTimeZone(Yii::$app->timeZone)))->format('Y-m-d H:i:s'),
                    'note' => $pay['note'] ?? null,
                ];
                $totalPaid += $amt;
            }

            if ($totalPaid <= 0) {
                return ['success'=>false,'message'=>'Tidak ada nominal pembayaran.'];
            }

            $db = Yii::$app->db;
            $transaction = $db->beginTransaction();
            try {
                $p = new Pembayaran();
                $p->nomor = $nomor;
                $p->pelanggan_id = $post['pelanggan_id'] ?? null;
                $p->kontrak_id = intval($post['kontrak_id'] ?? 0);
                $p->created_at = (new \DateTime('now', new \DateTimeZone(Yii::$app->timeZone)))->format('Y-m-d H:i:s');
                $p->total_bayar = $totalPaid;
                if (!$p->save()) {
                    $transaction->rollBack();
                    return ['success'=>false,'errors'=>$p->getErrors()];
                }

                foreach ($normalizedPayments as $np) {
                    $pi = new PembayaranItem();
                    $pi->pembayaran_id = $p->id;
                    $pi->kontrak_installment_id = $np['kontrak_installment_id'];
                    $pi->amount = $np['amount'];
                    $pi->paid_at = $np['paid_at'];
                    $pi->note = $np['note'];
                    if (!$pi->save()) {
                        $transaction->rollBack();
                        return ['success'=>false,'errors'=>$pi->getErrors()];
                    }
                }

                $transaction->commit();
                return ['success'=>true,'pembayaran_id'=>$p->id,'nomor'=>$p->nomor];
            } catch (\Throwable $e) {
                $transaction->rollBack();
                Yii::error('Save Pembayaran error: ' . $e->getMessage(), __METHOD__);
                return ['success'=>false,'message'=>$e->getMessage()];
            }
        }

        // GET: render create view
        return $this->render('create');
    }

    public function actionSearchPelanggan($q = null)
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        $out = [];
        if (!is_null($q)) {
            $q = trim($q);
            $rows = Pelanggan::find()->where(['or', ['like','nama',$q], ['like','no_ktp',$q], ['like','no_whatsapp',$q]])->limit(20)->all();
            foreach ($rows as $r) $out[] = ['id'=>$r->kode_pelanggan,'text'=>sprintf('%s — %s', $r->nama, $r->no_whatsapp ?: $r->no_handphone)];
        }
        return ['results'=>$out];
    }

    public function actionKontrakForPelanggan($pelanggan_id)
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        $out = [];
        $rows = Kontrak::find()->where(['pelanggan_id'=>$pelanggan_id])->all();
        foreach ($rows as $r) $out[] = ['id'=>$r->id,'text'=>sprintf('%s — %s', $r->nomor, Yii::$app->formatter->asDate($r->tanggal_mulai,'short'))];
        return $out;
    }

    public function actionGetInstallments($kontrak_id)
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        try {
            $installments = KontrakInstallment::find()->where(['kontrak_id'=>$kontrak_id])->orderBy(['periode'=>SORT_ASC])->all();
            // initial paid totals per installment (payments explicitly linked to installment)
            $paidTotals = [];
            $installmentIds = [];
            foreach ($installments as $ins) $installmentIds[] = $ins->id;
            if (!empty($installmentIds)) {
                $paidRows = (new \yii\db\Query())
                    ->select(['kontrak_installment_id','SUM(amount) AS total'])
                    ->from('pembayaran_item')
                    ->where(['kontrak_installment_id' => $installmentIds])
                    ->groupBy('kontrak_installment_id')
                    ->all();
                foreach ($paidRows as $pr) $paidTotals[$pr['kontrak_installment_id']] = floatval($pr['total']);
            }

            // gather orphan payments for this kontrak (pembayaran_item with no kontrak_installment_id)
            $orphanTotal = 0;
            $orphanRows = (new \yii\db\Query())
                ->select(['pi.amount'])
                ->from('pembayaran_item pi')
                ->leftJoin('pembayaran p','p.id = pi.pembayaran_id')
                ->where(['p.kontrak_id' => $kontrak_id])
                ->andWhere(['pi.kontrak_installment_id' => null])
                ->all();
            foreach ($orphanRows as $or) $orphanTotal += floatval($or['amount']);

            // allocate orphanTotal across installments in order
            foreach ($installments as $ins) {
                $due = floatval($ins->amount);
                $paid = $paidTotals[$ins->id] ?? 0;
                $remainingDue = max(0, $due - $paid);
                if ($remainingDue > 0 && $orphanTotal > 0) {
                    $alloc = min($remainingDue, $orphanTotal);
                    $paid += $alloc;
                    $orphanTotal -= $alloc;
                }
                $paidTotals[$ins->id] = $paid;
            }

            // build response rows
            $rows = [];
            foreach ($installments as $ins) {
                $rows[] = [
                    'id'=>$ins->id,
                    'periode'=>$ins->periode,
                    'due_date'=> Yii::$app->formatter->asDate($ins->due_date, 'medium'),
                    'due_date_raw' => $ins->due_date,
                    'amount'=>$ins->amount,
                    'paid'=> $paidTotals[$ins->id] ?? 0,
                ];
            }
            return $rows;
        } catch (\Throwable $e) {
            Yii::error("GetInstallments error: " . $e->getMessage(), __METHOD__);
            return ['error'=>true,'message'=>$e->getMessage()];
        }
    }

    public function actionView($id)
    {
        $model = Pembayaran::findOne($id);
        if (!$model) throw new \yii\web\NotFoundHttpException('Pembayaran tidak ditemukan');
        return $this->renderPartial('view',['model'=>$model]);
    }

    /**
     * Print a pembayaran receipt (openable in new tab)
     * @param int $id
     */
    public function actionPrint($id)
    {
        $model = Pembayaran::findOne($id);
        if (!$model) throw new \yii\web\NotFoundHttpException('Pembayaran tidak ditemukan');
        // reuse the view partial which contains the receipt markup
        return $this->renderPartial('view',['model'=>$model]);
    }

    /**
     * Show pembayaran history for a kontrak
     * @param int $kontrak_id
     */
    public function actionHistory($kontrak_id)
    {
        $query = Pembayaran::find()->where(['kontrak_id' => $kontrak_id])->orderBy(['created_at' => SORT_DESC]);
        $models = $query->all();

        // preload installments and pembayaran_item to avoid per-installment queries in view
        $kontrak = Kontrak::findOne($kontrak_id);
        $installments = $kontrak ? $kontrak->installments : [];
        $installmentIds = [];
        foreach ($installments as $ins) $installmentIds[] = $ins->id;

        $itemsByInstallment = [];
        if (!empty($installmentIds)) {
            // fetch pembayaran_item joined with pembayaran to order by real payment time (fallback to pembayaran.created_at)
            $rows = (new \yii\db\Query())
                ->select(['pi.*','p.created_at AS pembayaran_created_at'])
                ->from('pembayaran_item pi')
                ->leftJoin('pembayaran p','p.id = pi.pembayaran_id')
                ->where(['kontrak_installment_id' => $installmentIds])
                ->orderBy(new \yii\db\Expression("COALESCE(pi.paid_at, p.created_at) ASC"))
                ->all();
            foreach ($rows as $r) {
                $key = $r['kontrak_installment_id'] ?: 0;
                if (!isset($itemsByInstallment[$key])) $itemsByInstallment[$key] = [];
                $itemsByInstallment[$key][] = $r;
            }

            // also include pembayaran_item rows that belong to this kontrak's pembayaran but have no kontrak_installment_id
            $pembayaranIds = [];
            foreach ($models as $pm) $pembayaranIds[] = $pm->id;
            if (!empty($pembayaranIds)) {
                // orphan items: order by pembayaran created_at so we respect the order payments were created
                $orphanRows = (new \yii\db\Query())
                    ->select(['pi.*','p.created_at AS pembayaran_created_at'])
                    ->from('pembayaran_item pi')
                    ->leftJoin('pembayaran p','p.id = pi.pembayaran_id')
                    ->where(['pi.pembayaran_id' => $pembayaranIds])
                    ->andWhere(['pi.kontrak_installment_id' => null])
                    ->orderBy(new \yii\db\Expression('p.created_at ASC'))
                    ->all();

                // compute current paid totals per installment
                $paidTotals = [];
                foreach ($installments as $ins) {
                    $paidTotals[$ins->id] = 0;
                    if (isset($itemsByInstallment[$ins->id])) {
                        foreach ($itemsByInstallment[$ins->id] as $r) $paidTotals[$ins->id] += floatval($r['amount']);
                    }
                }

                // Try to allocate orphan amounts across installments (allow splitting a single payment across multiple periods)
                foreach ($orphanRows as $or) {
                    $remaining = floatval($or['amount']);
                    foreach ($installments as $ins) {
                        if ($remaining <= 0) break;
                        $insId = $ins->id;
                        $due = floatval($ins->amount);
                        $remainingDue = max(0, $due - ($paidTotals[$insId] ?? 0));
                        if ($remainingDue <= 0) continue;

                        $alloc = min($remaining, $remainingDue);
                        // create a shallow copy of the row and set allocated amount
                        $copy = $or;
                        $copy['amount'] = $alloc;
                        if (!isset($itemsByInstallment[$insId])) $itemsByInstallment[$insId] = [];
                        $itemsByInstallment[$insId][] = $copy;
                        $paidTotals[$insId] = ($paidTotals[$insId] ?? 0) + $alloc;
                        $remaining -= $alloc;
                    }
                    if ($remaining > 0) {
                        // leftover that couldn't be assigned to installments
                        $left = $or;
                        $left['amount'] = $remaining;
                        if (!isset($itemsByInstallment[0])) $itemsByInstallment[0] = [];
                        $itemsByInstallment[0][] = $left;
                    }
                }
            }
        }

        // build map of pembayaran created_at and nomor to fallback when item.paid_at is missing or has no time
        $pembayaranCreatedMap = [];
        $pembayaranNomorMap = [];
        $pembayaranTotalMap = [];
        foreach ($models as $pm) {
            $pembayaranCreatedMap[$pm->id] = $pm->created_at ?? null;
            $pembayaranNomorMap[$pm->id] = $pm->nomor ?? null;
            $pembayaranTotalMap[$pm->id] = $pm->total_bayar ?? null;
        }

        return $this->render('history', [
            'models' => $models,
            'kontrak_id' => $kontrak_id,
            'kontrak' => $kontrak,
            'installments' => $installments,
            'itemsByInstallment' => $itemsByInstallment,
            'pembayaranCreatedMap' => $pembayaranCreatedMap,
            'pembayaranNomorMap' => $pembayaranNomorMap,
            'pembayaranTotalMap' => $pembayaranTotalMap,
        ]);
    }
}
