<?php

namespace app\controllers;

use Yii;
use app\models\Pelanggan;
use yii\data\ActiveDataProvider;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\filters\VerbFilter;
use yii\web\ForbiddenHttpException;
use yii\web\UploadedFile;
use app\components\ImageHelper;
use yii\web\BadRequestHttpException;
use yii\web\Response;
use yii\helpers\Url;

class PelangganController extends Controller
{
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['POST'],
                    'delete-photo' => ['POST'],
                ],
            ],
        ];
    }

    public function beforeAction($action)
    {
        if (!parent::beforeAction($action)) return false;
        $map = [
            'index' => 'pelanggan.view',
            'view' => 'pelanggan.view',
            'create' => 'pelanggan.create',
            'update' => 'pelanggan.update',
            'delete' => 'pelanggan.delete',
            'delete-photo' => 'pelanggan.update',
        ];
        $id = $action->id;
        $required = $map[$id] ?? null;
        if ($required) {
            if (Yii::$app->user->isGuest || !Yii::$app->user->identity->hasPermission($required)) {
                throw new ForbiddenHttpException('Anda tidak memiliki izin untuk mengakses halaman ini.');
            }
        }
        return true;
    }

    public function actionIndex()
    {
        $perPage = (int)Yii::$app->request->get('per-page', 15);
        $allowed = [15,25,50,100];
        if (!in_array($perPage, $allowed)) {
            $perPage = 15;
        }
        $query = Pelanggan::find();

        // combined search parameter 'q' searches across kode_pelanggan, nama, no_ktp
        $q = Yii::$app->request->get('q');
        if ($q !== null && $q !== '') {
            $query->andFilterWhere(['or',
                ['like', 'kode_pelanggan', $q],
                ['like', 'nama', $q],
                ['like', 'no_ktp', $q],
            ]);
        }

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => ['pageSize' => $perPage],
        ]);

        return $this->render('index', [
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionView($kode_pelanggan)
    {
        $model = $this->findModel($kode_pelanggan);
        if (Yii::$app->request->isAjax) {
            return $this->renderAjax('view', ['model' => $model]);
        }
        return $this->render('view', ['model' => $model]);
    }

    public function actionCreate()
    {
        $model = new Pelanggan();
        // pre-generate kode_pelanggan so the create form shows the auto-generated value
        try {
            $model->kode_pelanggan = $model->generateKode();
        } catch (\Throwable $e) {
            // if generation fails for any reason, leave blank and controller will generate before save
        }
        if ($model->load(Yii::$app->request->post())) {
            $model->fotoFile = UploadedFile::getInstance($model, 'fotoFile');
            $model->fotoKtpFile = UploadedFile::getInstance($model, 'fotoKtpFile');

            $dir = Yii::getAlias('@webroot/uploads/pelanggan');
            if (!is_dir($dir)) {
                mkdir($dir, 0755, true);
            }

            // foto
            if ($model->fotoFile) {
                $filename = uniqid() . '.' . $model->fotoFile->extension;
                if ($model->fotoFile->saveAs($dir . DIRECTORY_SEPARATOR . $filename)) {
                    $model->foto = $filename;
                    $thumbDir = $dir . DIRECTORY_SEPARATOR . 'thumbs';
                    if (!is_dir($thumbDir)) {
                        mkdir($thumbDir, 0755, true);
                    }
                    ImageHelper::createThumbnail($dir . DIRECTORY_SEPARATOR . $filename, $thumbDir . DIRECTORY_SEPARATOR . $filename, 300, 300);
                }
            }

            // foto_ktp
            if ($model->fotoKtpFile) {
                $filenameKtp = uniqid('ktp_') . '.' . $model->fotoKtpFile->extension;
                if ($model->fotoKtpFile->saveAs($dir . DIRECTORY_SEPARATOR . $filenameKtp)) {
                    $model->foto_ktp = $filenameKtp;
                    $thumbDir = $dir . DIRECTORY_SEPARATOR . 'thumbs';
                    if (!is_dir($thumbDir)) {
                        mkdir($thumbDir, 0755, true);
                    }
                    ImageHelper::createThumbnail($dir . DIRECTORY_SEPARATOR . $filenameKtp, $thumbDir . DIRECTORY_SEPARATOR . $filenameKtp, 300, 300);
                }
            }

            // multiple foto_lainnya
            $instances = UploadedFile::getInstances($model, 'fotoLainnyaFiles');
            if (!empty($instances)) {
                $saved = [];
                foreach ($instances as $inst) {
                    $fn = uniqid() . '.' . $inst->extension;
                    if ($inst->saveAs($dir . DIRECTORY_SEPARATOR . $fn)) {
                        $saved[] = $fn;
                        $thumbDir = $dir . DIRECTORY_SEPARATOR . 'thumbs';
                        if (!is_dir($thumbDir)) {
                            mkdir($thumbDir, 0755, true);
                        }
                        ImageHelper::createThumbnail($dir . DIRECTORY_SEPARATOR . $fn, $thumbDir . DIRECTORY_SEPARATOR . $fn, 300, 300);
                    }
                }
                if (!empty($saved)) {
                    $existing = [];
                    if (!empty($model->foto_lainnya)) {
                        $existing = explode('|', $model->foto_lainnya);
                    }
                    $all = array_merge($existing, $saved);
                    $model->foto_lainnya = implode('|', $all);
                }
            }

            // ensure kode_pelanggan is generated before saving
            if (empty($model->kode_pelanggan)) {
                $model->kode_pelanggan = $model->generateKode();
            }
            if ($model->save(false)) {
                Yii::$app->session->setFlash('success', 'Pelanggan berhasil dibuat.');
                if (Yii::$app->request->isAjax) {
                    Yii::$app->response->format = Response::FORMAT_JSON;
                    return ['success' => true, 'redirect' => Url::to(['index'])];
                }
                return $this->redirect(['index']);
            }
        }

        if (Yii::$app->request->isAjax) {
            return $this->renderAjax('create', ['model' => $model]);
        }

        return $this->render('create', ['model' => $model]);
    }

    public function actionUpdate($kode_pelanggan)
    {
        $model = $this->findModel($kode_pelanggan);

        if ($model->load(Yii::$app->request->post())) {
            // collect uploaded files but don't move them yet
            $model->fotoFile = UploadedFile::getInstance($model, 'fotoFile');
            $model->fotoKtpFile = UploadedFile::getInstance($model, 'fotoKtpFile');

            // validate first to allow file validator to inspect the temporary upload
            if ($model->validate()) {
                $dir = Yii::getAlias('@webroot/uploads/pelanggan');
                if (!is_dir($dir)) {
                    mkdir($dir, 0755, true);
                }

                if ($model->fotoFile) {
                    $filename = uniqid() . '.' . $model->fotoFile->extension;
                    if ($model->fotoFile->saveAs($dir . DIRECTORY_SEPARATOR . $filename)) {
                        if (!empty($model->getOldAttribute('foto'))) {
                            $old = $dir . DIRECTORY_SEPARATOR . $model->getOldAttribute('foto');
                            if (is_file($old)) {
                                @unlink($old);
                            }
                        }
                        $model->foto = $filename;
                    }
                }

                if ($model->fotoKtpFile) {
                    $filenameKtp = uniqid('ktp_') . '.' . $model->fotoKtpFile->extension;
                    if ($model->fotoKtpFile->saveAs($dir . DIRECTORY_SEPARATOR . $filenameKtp)) {
                        if (!empty($model->getOldAttribute('foto_ktp'))) {
                            $old = $dir . DIRECTORY_SEPARATOR . $model->getOldAttribute('foto_ktp');
                            if (is_file($old)) {
                                @unlink($old);
                            }
                        }
                        $model->foto_ktp = $filenameKtp;
                    }
                }

                // multiple foto_lainnya (append new files)
                $instances = UploadedFile::getInstances($model, 'fotoLainnyaFiles');
                if (!empty($instances)) {
                    $saved = [];
                    foreach ($instances as $inst) {
                        $fn = uniqid() . '.' . $inst->extension;
                        if ($inst->saveAs($dir . DIRECTORY_SEPARATOR . $fn)) {
                            $saved[] = $fn;
                            $thumbDir = $dir . DIRECTORY_SEPARATOR . 'thumbs';
                            if (!is_dir($thumbDir)) {
                                mkdir($thumbDir, 0755, true);
                            }
                            ImageHelper::createThumbnail($dir . DIRECTORY_SEPARATOR . $fn, $thumbDir . DIRECTORY_SEPARATOR . $fn, 300, 300);
                        }
                    }
                    if (!empty($saved)) {
                        $existing = [];
                        $oldVal = $model->getOldAttribute('foto_lainnya');
                        if (!empty($oldVal)) {
                            $existing = explode('|', $oldVal);
                        }
                        $all = array_merge($existing, $saved);
                        $model->foto_lainnya = implode('|', $all);
                    }
                }

                if ($model->save(false)) {
                    Yii::$app->session->setFlash('success', 'Pelanggan berhasil diubah.');
                    if (Yii::$app->request->isAjax) {
                        Yii::$app->response->format = Response::FORMAT_JSON;
                        return ['success' => true, 'redirect' => Url::to(['index'])];
                    }
                    return $this->redirect(['index']);
                }
            }
        }

        if (Yii::$app->request->isAjax) {
            return $this->renderAjax('update', ['model' => $model]);
        }

        return $this->render('update', ['model' => $model]);
    }

    public function actionDelete($kode_pelanggan)
    {
        $this->findModel($kode_pelanggan)->delete();
        if (Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ['success' => true];
        }
        Yii::$app->session->setFlash('success', 'Pelanggan berhasil dihapus.');
        return $this->redirect(['index']);
    }

    /**
     * Deletes a single image file (foto, foto_ktp or one of foto_lainnya) for a pelanggan.
     * Expects POST with 'file' (the filename or relative uploads path) and optional 'redirect'.
     */
    public function actionDeletePhoto($kode_pelanggan)
    {
        $model = $this->findModel($kode_pelanggan);
        $file = Yii::$app->request->post('file');
        $isAjax = Yii::$app->request->isAjax;
        $isPost = Yii::$app->request->isPost;
        // For AJAX and also for direct POST requests, respond with JSON to simplify client handling.
        $respondJson = $isAjax || $isPost;
        if (empty($file)) {
            if ($respondJson) {
                return $this->asJson(['success' => false, 'message' => 'Missing file parameter.']);
            }
            throw new BadRequestHttpException('Missing file parameter.');
        }

        // normalize incoming value
        $file = trim($file);
        $basename = basename($file);
        $webroot = Yii::getAlias('@webroot');
        $dir = $webroot . DIRECTORY_SEPARATOR . 'uploads' . DIRECTORY_SEPARATOR . 'pelanggan';

        // Helper to unlink both original and thumb
        $unlinkFile = function($name){
            $webroot = Yii::getAlias('@webroot');
            $dir = $webroot . DIRECTORY_SEPARATOR . 'uploads' . DIRECTORY_SEPARATOR . 'pelanggan';
            $orig = $dir . DIRECTORY_SEPARATOR . $name;
            $thumb = $dir . DIRECTORY_SEPARATOR . 'thumbs' . DIRECTORY_SEPARATOR . $name;
            if (is_file($orig)) {@unlink($orig);} 
            if (is_file($thumb)) {@unlink($thumb);} 
        };

        // 1) If matches main foto
        if (!empty($model->foto) && (basename($model->foto) === $basename || $model->foto === $file || ltrim($model->foto, '/') === ltrim($file, '/'))) {
            $unlinkFile(basename($model->foto));
            $model->foto = null;
            $model->save(false);
            if ($respondJson) {
                return $this->asJson(['success' => true, 'message' => 'Foto utama dihapus.']);
            }
            Yii::$app->session->setFlash('success', 'Foto utama dihapus.');
            return $this->redirect(Yii::$app->request->post('redirect', ['update','kode_pelanggan'=>$model->kode_pelanggan]));
        }

        // 2) If matches foto_ktp
        if (!empty($model->foto_ktp) && (basename($model->foto_ktp) === $basename || $model->foto_ktp === $file || ltrim($model->foto_ktp, '/') === ltrim($file, '/'))) {
            $unlinkFile(basename($model->foto_ktp));
            $model->foto_ktp = null;
            $model->save(false);
            if ($respondJson) {
                return $this->asJson(['success' => true, 'message' => 'Foto KTP dihapus.']);
            }
            Yii::$app->session->setFlash('success', 'Foto KTP dihapus.');
            return $this->redirect(Yii::$app->request->post('redirect', ['update','kode_pelanggan'=>$model->kode_pelanggan]));
        }

        // 3) Try foto_lainnya list
        if (!empty($model->foto_lainnya)) {
            $parts = explode('|', $model->foto_lainnya);
            $new = [];
            $deleted = false;
            foreach ($parts as $p) {
                if (!$p) continue;
                // compare by basename or exact match
                if (basename($p) === $basename || $p === $file || ltrim($p, '/') === ltrim($file, '/')) {
                    // delete this file
                    $unlinkFile(basename($p));
                    $deleted = true;
                    continue;
                }
                $new[] = $p;
            }
            if ($deleted) {
                $model->foto_lainnya = !empty($new) ? implode('|', $new) : null;
                $model->save(false);
                if ($respondJson) {
                    return $this->asJson(['success' => true, 'message' => 'Gambar berhasil dihapus.']);
                }
                Yii::$app->session->setFlash('success', 'Gambar berhasil dihapus.');
                return $this->redirect(Yii::$app->request->post('redirect', ['update','kode_pelanggan'=>$model->kode_pelanggan]));
            }
        }

        if ($respondJson) {
            return $this->asJson(['success' => false, 'message' => 'File tidak ditemukan atau tidak dapat dihapus.']);
        }
        Yii::$app->session->setFlash('warning', 'File tidak ditemukan atau tidak dapat dihapus.');
        return $this->redirect(Yii::$app->request->post('redirect', ['update','kode_pelanggan'=>$model->kode_pelanggan]));
    }

    protected function findModel($id)
    {
        if (($model = Pelanggan::findOne(['kode_pelanggan' => $id])) !== null) {
            return $model;
        }
        throw new NotFoundHttpException('The requested page does not exist.');
    }
}
