<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\web\Response;
use yii\filters\VerbFilter;
use yii\web\ForbiddenHttpException;
use app\models\Kontrak;
use app\models\KontrakItem;
use app\models\KontrakInstallment;
use app\models\Pelanggan;
use app\models\Barang;
use app\models\Setting;

class KontrakController extends Controller
{
    public function behaviors()
    {
        return [
            'verbs'=>[
                'class'=>VerbFilter::class,
                'actions'=>[
                    'create'=>['GET','POST'],
                    'search-pelanggan'=>['GET'],
                    'search-barang'=>['GET'],
                    'create-pelanggan'=>['POST'],
                    'delete'=>['POST'],
                ],
            ],
        ];
    }

    public function beforeAction($action)
    {
        if (!parent::beforeAction($action)) return false;
        $map = [
            'index' => 'kontrak.view',
            'view' => 'kontrak.view',
            'create' => 'kontrak.create',
            'delete' => 'kontrak.delete',
            'print' => 'kontrak.print',
            'search-pelanggan' => 'kontrak.view',
            'search-barang' => 'kontrak.view',
            'create-pelanggan' => 'kontrak.create',
        ];
        $id = $action->id;
        $required = $map[$id] ?? null;
        if ($required) {
            if (Yii::$app->user->isGuest || !Yii::$app->user->identity->hasPermission($required)) {
                throw new ForbiddenHttpException('Anda tidak memiliki izin untuk mengakses halaman ini.');
            }
        }
        return true;
    }

    public function actionCreate()
    {
        $request = Yii::$app->request;
        if ($request->isPost) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            $post = $request->post();

            $pelanggan_id = $post['pelanggan_id'] ?? null;
            $pelanggan_data = $post['pelanggan_new'] ?? null; // if creating

            if (empty($pelanggan_id) && !empty($pelanggan_data)) {
                $p = new Pelanggan();
                $p->attributes = $pelanggan_data;
                if (empty($p->kode_pelanggan)) $p->kode_pelanggan = $p->generateKode();
                if (!$p->save()) {
                    return ['success'=>false,'errors'=>$p->getErrors()];
                }
                $pelanggan_id = $p->kode_pelanggan;
            }

            if (empty($pelanggan_id)) {
                return ['success'=>false,'message'=>'Pelanggan belum dipilih'];
            }

            $items = $post['items'] ?? [];
            if (empty($items)) {
                return ['success'=>false,'message'=>'Minimal 1 barang harus dipilih'];
            }

            // compute total harga
            $total = 0.0;
            foreach ($items as $it) {
                $price = floatval($it['harga'] ?? 0);
                $qty = intval($it['qty'] ?? 1);
                $total += $price * $qty;
            }

            $dp = floatval($post['dp'] ?? 0);
            $margin = floatval($post['margin_per_week'] ?? 2.0);
            $tenor = intval($post['tenor'] ?? 10);
            $tenor_unit = ($post['tenor_unit'] ?? 'week') === 'month' ? 'month' : 'week';
            $start_date = $post['tanggal_mulai'] ?? date('Y-m-d');
            $is_custom = !empty($post['is_custom_installment']) ? 1 : 0;
            $custom_amount = $is_custom ? floatval($post['custom_installment_amount'] ?? 0) : null;

            // generate nomor
            $prefix = Setting::get('kontrak_prefix','KTR');
            $today = date('Ymd');
            $seqKey = 'kontrak_seq_'.$today;
            $seq = intval(Setting::get($seqKey,0)) + 1;
            Setting::set($seqKey,$seq);
            $nomor = sprintf('%s-%s-%04d', $prefix, date('ymd'), $seq);

            $kontrak = new Kontrak();
            $kontrak->nomor = $nomor;
            $kontrak->pelanggan_id = $pelanggan_id;
            $kontrak->tanggal_mulai = $start_date;
            $kontrak->dp = $dp;
            $kontrak->margin_per_week = $margin;
            $kontrak->tenor = $tenor;
            $kontrak->tenor_unit = $tenor_unit;
            $kontrak->total_harga = $total;
            $kontrak->is_custom_installment = $is_custom;
            $kontrak->custom_installment_amount = $custom_amount;
            $kontrak->created_at = date('Y-m-d H:i:s');
            if (!$kontrak->save()) {
                return ['success'=>false,'errors'=>$kontrak->getErrors()];
            }

            foreach ($items as $it) {
                $ki = new KontrakItem();
                $ki->kontrak_id = $kontrak->id;
                $ki->barang_id = $it['barang_id'];
                $ki->qty = intval($it['qty']);
                $ki->harga = floatval($it['harga']);
                $ki->keterangan = $it['keterangan'] ?? null;
                $ki->save(false);
            }

            // schedule installments
            $remaining = $total - $dp;
            $tenor_weeks = $tenor_unit === 'week' ? $tenor : ($tenor * 4);
            $total_with_interest = $remaining + ($remaining * ($margin/100.0) * $tenor_weeks);
            $periods = $tenor;
            if ($is_custom && $custom_amount > 0) {
                $amt = $custom_amount;
                for ($p=1;$p<=$periods;$p++) {
                    $due = $this->computeDueDate($start_date,$p,$tenor_unit);
                    $ins = new KontrakInstallment();
                    $ins->kontrak_id = $kontrak->id;
                    $ins->periode = $p;
                    $ins->due_date = $due;
                    $ins->amount = $amt;
                    $ins->save(false);
                }
            } else {
                $raw = $total_with_interest / $periods;
                $rounded = $this->roundToNearestThousand($raw);
                for ($p=1;$p<=$periods;$p++) {
                    $due = $this->computeDueDate($start_date,$p,$tenor_unit);
                    $ins = new KontrakInstallment();
                    $ins->kontrak_id = $kontrak->id;
                    $ins->periode = $p;
                    $ins->due_date = $due;
                    $ins->amount = $rounded;
                    $ins->save(false);
                }
            }

            return ['success'=>true,'kontrak_id'=>$kontrak->id,'nomor'=>$kontrak->nomor];
        }

        // GET: render the create page
        return $this->render('create');
    }

    protected function roundToNearestThousand($value)
    {
        return round($value/1000)*1000;
    }

    protected function computeDueDate($start,$periodIndex,$unit)
    {
        $dt = new \DateTime($start);
        if ($unit === 'week') {
            $interval = new \DateInterval('P'.($periodIndex).'W');
            $dt->add($interval);
        } else {
            $interval = new \DateInterval('P'.($periodIndex).'M');
            $dt->add($interval);
        }
        return $dt->format('Y-m-d');
    }

    public function actionSearchPelanggan($q = null)
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        $out = [];
        if (!is_null($q)) {
            $q = trim($q);
            $rows = Pelanggan::find()->where(['or',
                ['like','nama',$q],
                ['like','no_ktp',$q],
                ['like','no_whatsapp',$q],
                ['like','no_handphone',$q],
            ])->limit(20)->all();
            foreach ($rows as $r) {
                $txt = sprintf('%s — %s / %s', $r->nama, $r->no_ktp, $r->no_whatsapp ?: $r->no_handphone);
                $out[] = ['id'=>$r->kode_pelanggan,'text'=>$txt];
            }
        }
        return ['results'=>$out];
    }

    public function actionSearchBarang($q = null)
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        $out = [];
        if (!is_null($q)) {
            $q = trim($q);
            $rows = Barang::find()->where(['or',
                ['like','nama',$q],
                ['like','deskripsi',$q],
                ['like','kode_barang',$q],
            ])->limit(50)->all();
            foreach ($rows as $r) {
                $out[] = ['id'=>$r->kode_barang,'text'=>sprintf('%s — %s (Rp %s)',$r->nama,$r->deskripsi,number_format($r->harga,0,',','.')),'harga'=>floatval($r->harga),'kode'=>$r->kode_barang];
            }
        }
        return ['results'=>$out];
    }

    public function actionCreatePelanggan()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        $post = Yii::$app->request->post();
        $p = new Pelanggan();
        // map possible incoming fields
        if (isset($post['nama'])) $p->nama = $post['nama'];
        if (isset($post['no_ktp'])) $p->no_ktp = $post['no_ktp'];
        if (isset($post['no_wa'])) $p->no_whatsapp = $post['no_wa'];
        if (empty($p->kode_pelanggan)) $p->kode_pelanggan = $p->generateKode();
        if (empty($p->created_at)) $p->created_at = date('Y-m-d H:i:s');
        if ($p->save()) {
            return ['success'=>true,'kode'=>$p->kode_pelanggan,'nama'=>$p->nama];
        }
        return ['success'=>false,'errors'=>$p->getErrors()];
    }

    public function actionIndex()
    {
        $request = Yii::$app->request;
        $q = trim($request->get('q',''));
        $from = $request->get('from','');
        $to = $request->get('to','');

        // Use explicit LEFT JOIN with COLLATE to avoid mixed-collation errors between kontrak.pelanggan_id and pelanggan.kode_pelanggan
        $query = Kontrak::find()->alias('k')
            ->leftJoin('pelanggan p', "k.pelanggan_id COLLATE utf8mb4_general_ci = p.kode_pelanggan COLLATE utf8mb4_general_ci");

        if ($q !== '') {
            $query->andWhere(['or',
                ['like','k.nomor',$q],
                ['like','p.nama',$q],
                ['like','p.no_whatsapp',$q],
                ['like','p.no_handphone',$q],
            ]);
        }

        if ($from !== '' || $to !== '') {
            if ($from !== '' && $to !== '') {
                $query->andWhere(['between','k.tanggal_mulai',$from,$to]);
            } elseif ($from !== '') {
                $query->andWhere(['>=','k.tanggal_mulai',$from]);
            } elseif ($to !== '') {
                $query->andWhere(['<=','k.tanggal_mulai',$to]);
            }
        } else {
            // default: show today's contracts
            $today = date('Y-m-d');
            $query->andWhere(['k.tanggal_mulai' => $today]);
        }

        $query->orderBy(['k.tanggal_mulai'=>SORT_DESC,'k.created_at'=>SORT_DESC]);

        $countQuery = clone $query;
        $pagination = new \yii\data\Pagination(['totalCount'=>$countQuery->count(),'pageSize'=>20]);
        $models = $query->offset($pagination->offset)->limit($pagination->limit)->all();

        // if AJAX request (pagination or filter), return only the list partial
        if (Yii::$app->request->isAjax) {
            return $this->renderPartial('_list',['models'=>$models,'pagination'=>$pagination,'q'=>$q,'from'=>$from,'to'=>$to]);
        }

        return $this->render('index',['models'=>$models,'pagination'=>$pagination,'q'=>$q,'from'=>$from,'to'=>$to]);
    }

    public function actionDelete($id)
    {
        $model = Kontrak::findOne($id);
        if ($model) {
            // delete related items and installments
            KontrakItem::deleteAll(['kontrak_id'=>$model->id]);
            KontrakInstallment::deleteAll(['kontrak_id'=>$model->id]);
            $model->delete();
            Yii::$app->session->setFlash('success','Kontrak dihapus.');
        } else {
            Yii::$app->session->setFlash('error','Kontrak tidak ditemukan.');
        }
        return $this->redirect(['index']);
    }

    public function actionPrint($id)
    {
        $model = Kontrak::findOne($id);
        if (!$model) throw new \yii\web\NotFoundHttpException('Kontrak tidak ditemukan');
        return $this->renderPartial('print', ['model'=>$model]);
    }

    public function actionPrintSchedule($id)
    {
        $model = Kontrak::findOne($id);
        if (!$model) throw new \yii\web\NotFoundHttpException('Kontrak tidak ditemukan');
        $installments = $model->installments;
        return $this->renderPartial('print_schedule', ['model'=>$model,'installments'=>$installments]);
    }

    public function actionView($id)
    {
        $model = Kontrak::findOne($id);
        if (!$model) throw new \yii\web\NotFoundHttpException('Kontrak tidak ditemukan');
        $items = $model->items;
        $installments = $model->installments;
        return $this->renderPartial('_detail', ['model'=>$model,'items'=>$items,'installments'=>$installments]);
    }
}
