<?php

namespace app\controllers;

use Yii;
use app\models\Barang;
use app\models\Kategori;
use yii\data\ActiveDataProvider;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\filters\VerbFilter;
use yii\web\ForbiddenHttpException;
use yii\web\UploadedFile;
use app\components\ImageHelper;
use yii\web\Response;
use yii\helpers\Url;

class BarangController extends Controller
{
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
        ];
    }

    public function beforeAction($action)
    {
        if (!parent::beforeAction($action)) return false;
        $map = [
            'index' => 'barang.view',
            'view' => 'barang.view',
            'create' => 'barang.create',
            'update' => 'barang.update',
            'delete' => 'barang.delete',
            'next-kode' => 'barang.create',
        ];
        $id = $action->id;
        $required = $map[$id] ?? null;
        if ($required) {
            if (Yii::$app->user->isGuest || !Yii::$app->user->identity->hasPermission($required)) {
                throw new ForbiddenHttpException('Anda tidak memiliki izin untuk mengakses halaman ini.');
            }
        }
        return true;
    }

    public function actionIndex()
    {
        $perPage = (int)Yii::$app->request->get('per-page', 15);
        $allowed = [15,25,50,100];
        if (!in_array($perPage, $allowed)) {
            $perPage = 15;
        }
        $query = Barang::find();

        // combined search parameter 'q' searches across kode_barang, nama, deskripsi
        $q = Yii::$app->request->get('q');
        if ($q !== null && $q !== '') {
            $query->andFilterWhere(['or',
                ['like', 'kode_barang', $q],
                ['like', 'nama', $q],
                ['like', 'deskripsi', $q],
            ]);
        }

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => ['pageSize' => $perPage],
        ]);

        return $this->render('index', [
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionView($id)
    {
        $model = $this->findModel($id);
        if (Yii::$app->request->isAjax) {
            return $this->renderAjax('view', ['model' => $model]);
        }
        return $this->render('view', ['model' => $model]);
    }

    public function actionCreate()
    {
        $model = new Barang();

        if ($model->load(Yii::$app->request->post())) {
            // if kategori was submitted as an ID, convert to name
            $post = Yii::$app->request->post('Barang');
            if (!empty($post['kategori']) && is_numeric($post['kategori'])) {
                $kat = Kategori::findOne($post['kategori']);
                if ($kat) {
                    $model->kategori = $kat->nama;
                }
            }
            $model->fotoFile = UploadedFile::getInstance($model, 'fotoFile');
            if ($model->fotoFile) {
                $dir = Yii::getAlias('@webroot/uploads/barang');
                if (!is_dir($dir)) {
                    mkdir($dir, 0755, true);
                }
                $filename = uniqid() . '.' . $model->fotoFile->extension;
                if ($model->fotoFile->saveAs($dir . DIRECTORY_SEPARATOR . $filename)) {
                    $model->foto = $filename;
                    // create thumbnail
                    $thumbDir = $dir . DIRECTORY_SEPARATOR . 'thumbs';
                    if (!is_dir($thumbDir)) {
                        mkdir($thumbDir, 0755, true);
                    }
                    ImageHelper::createThumbnail($dir . DIRECTORY_SEPARATOR . $filename, $thumbDir . DIRECTORY_SEPARATOR . $filename, 300, 300);
                }
            }

            // generate kode_barang from existing values (no autoincrement id available)
            if (empty($model->kode_barang)) {
                $model->kode_barang = $model->generateKode();
            }
            if ($model->save(false)) {
                Yii::$app->session->setFlash('success', 'Barang berhasil dibuat.');
                if (Yii::$app->request->isAjax) {
                    Yii::$app->response->format = Response::FORMAT_JSON;
                    return ['success' => true, 'redirect' => Url::to(['index'])];
                }
                return $this->redirect(['index']);
            }
        }

        if (Yii::$app->request->isAjax) {
            return $this->renderAjax('create', ['model' => $model]);
        }

        return $this->render('create', ['model' => $model]);
    }

    public function actionUpdate($id)
    {
        $model = $this->findModel($id);

        if ($model->load(Yii::$app->request->post())) {
            // if kategori submitted as ID, convert to name
            $post = Yii::$app->request->post('Barang');
            if (!empty($post['kategori']) && is_numeric($post['kategori'])) {
                $kat = Kategori::findOne($post['kategori']);
                if ($kat) {
                    $model->kategori = $kat->nama;
                }
            }
            $model->fotoFile = UploadedFile::getInstance($model, 'fotoFile');
            if ($model->fotoFile) {
                $dir = Yii::getAlias('@webroot/uploads/barang');
                if (!is_dir($dir)) {
                    mkdir($dir, 0755, true);
                }
                $filename = uniqid() . '.' . $model->fotoFile->extension;
                if ($model->fotoFile->saveAs($dir . DIRECTORY_SEPARATOR . $filename)) {
                    // remove old file and old thumb if exist
                    if (!empty($model->getOldAttribute('foto'))) {
                        $old = $dir . DIRECTORY_SEPARATOR . $model->getOldAttribute('foto');
                        if (is_file($old)) {
                            @unlink($old);
                        }
                        $oldThumb = $dir . DIRECTORY_SEPARATOR . 'thumbs' . DIRECTORY_SEPARATOR . $model->getOldAttribute('foto');
                        if (is_file($oldThumb)) {
                            @unlink($oldThumb);
                        }
                    }
                    $model->foto = $filename;
                    $thumbDir = $dir . DIRECTORY_SEPARATOR . 'thumbs';
                    if (!is_dir($thumbDir)) {
                        mkdir($thumbDir, 0755, true);
                    }
                    ImageHelper::createThumbnail($dir . DIRECTORY_SEPARATOR . $filename, $thumbDir . DIRECTORY_SEPARATOR . $filename, 300, 300);
                }
            }

            if ($model->save(false)) {
                if (empty($model->kode_barang)) {
                    $model->kode_barang = $model->generateKode();
                    $model->save(false);
                }
                Yii::$app->session->setFlash('success', 'Barang berhasil diperbarui.');
                if (Yii::$app->request->isAjax) {
                    Yii::$app->response->format = Response::FORMAT_JSON;
                    return ['success' => true, 'redirect' => Url::to(['index'])];
                }
                return $this->redirect(['index']);
            }
        }

        if (Yii::$app->request->isAjax) {
            return $this->renderAjax('update', ['model' => $model]);
        }

        return $this->render('update', ['model' => $model]);
    }

    /**
     * Return a predicted next kode for barang using prefix and next id (max+1)
     * Accessible via GET `index.php?r=barang/next-kode`
     */
    public function actionNextKode()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        // Use model generator to compute next kode (works without id auto-increment)
        $model = new Barang();
        $kode = $model->generateKode();
        return ['kode' => $kode];
    }

    public function actionDelete($id)
    {
        $this->findModel($id)->delete();
        if (Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ['success' => true];
        }
        Yii::$app->session->setFlash('success', 'Barang berhasil dihapus.');
        return $this->redirect(['index']);
    }

    protected function findModel($id)
    {
        if (($model = Barang::findOne($id)) !== null) {
            return $model;
        }
        throw new NotFoundHttpException('The requested page does not exist.');
    }
}
