<?php

namespace app\components;

class ImageHelper
{
    /**
     * Create a thumbnail for an image using GD.
     * @param string $srcPath
     * @param string $destPath
     * @param int $maxWidth
     * @param int $maxHeight
     * @return bool
     */
    public static function createThumbnail(string $srcPath, string $destPath, int $maxWidth = 300, int $maxHeight = 300): bool
    {
        if (!is_file($srcPath)) {
            return false;
        }

        [$width, $height, $type] = getimagesize($srcPath);
        if (!$width || !$height) {
            return false;
        }

        $ratio = min($maxWidth / $width, $maxHeight / $height);
        $newWidth = (int) max(1, floor($width * $ratio));
        $newHeight = (int) max(1, floor($height * $ratio));

        switch ($type) {
            case IMAGETYPE_JPEG:
                $srcImg = imagecreatefromjpeg($srcPath);
                break;
            case IMAGETYPE_PNG:
                $srcImg = imagecreatefrompng($srcPath);
                break;
            case IMAGETYPE_GIF:
                $srcImg = imagecreatefromgif($srcPath);
                break;
            default:
                return false;
        }

        $thumb = imagecreatetruecolor($newWidth, $newHeight);

        // Preserve PNG alpha
        if ($type === IMAGETYPE_PNG) {
            imagealphablending($thumb, false);
            imagesavealpha($thumb, true);
            $transparent = imagecolorallocatealpha($thumb, 255, 255, 255, 127);
            imagefilledrectangle($thumb, 0, 0, $newWidth, $newHeight, $transparent);
        }

        imagecopyresampled($thumb, $srcImg, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);

        $saved = false;
        switch ($type) {
            case IMAGETYPE_JPEG:
                $saved = imagejpeg($thumb, $destPath, 85);
                break;
            case IMAGETYPE_PNG:
                $saved = imagepng($thumb, $destPath);
                break;
            case IMAGETYPE_GIF:
                $saved = imagegif($thumb, $destPath);
                break;
        }

        imagedestroy($srcImg);
        imagedestroy($thumb);

        return (bool) $saved;
    }
}
